<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Filiere;
use Illuminate\Http\Request;

class FiliereController extends Controller
{
    public function index()
    {
        $filieres = Filiere::ordre()->with('niveaux')->get();
        return view('admin.filieres.index', compact('filieres'));
    }

    public function create()
    {
        return view('admin.filieres.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nom' => 'required|string|max:255',
            'description' => 'nullable|string',
            'statut' => 'boolean',
            'ordre' => 'nullable|integer',
            'nombre_niveaux' => 'required|integer|min:1|max:5',
        ]);

        $validated['statut'] = $request->has('statut');
        $validated['ordre'] = $validated['ordre'] ?? Filiere::max('ordre') + 1;

        $nombreNiveaux = $validated['nombre_niveaux'];
        unset($validated['nombre_niveaux']);

        $filiere = Filiere::create($validated);

        // Creer automatiquement les niveaux selon le nombre choisi
        $niveaux = [];
        for ($i = 1; $i <= $nombreNiveaux; $i++) {
            $niveaux[] = ['nom' => 'Niveau ' . $i, 'statut' => true];
        }
        $filiere->niveaux()->createMany($niveaux);

        return redirect()->route('admin.filieres.configure', $filiere)
            ->with('success', 'Filiere creee avec succes. Configurez maintenant les matieres, frais et documents pour chaque niveau.');
    }

    public function edit(Filiere $filiere)
    {
        return view('admin.filieres.edit', compact('filiere'));
    }

    public function update(Request $request, Filiere $filiere)
    {
        $validated = $request->validate([
            'nom' => 'required|string|max:255',
            'description' => 'nullable|string',
            'statut' => 'boolean',
            'ordre' => 'nullable|integer',
        ]);

        $validated['statut'] = $request->has('statut');

        $filiere->update($validated);

        return redirect()->route('admin.filieres.index')
            ->with('success', 'Filiere mise a jour avec succes.');
    }

    public function destroy(Filiere $filiere)
    {
        $filiere->delete();

        return redirect()->route('admin.filieres.index')
            ->with('success', 'Filiere supprimee avec succes.');
    }

    public function show(Filiere $filiere)
    {
        $filiere->load(['niveaux.matieres', 'niveaux.frais', 'niveaux.documentsNiveau']);
        return view('admin.filieres.show', compact('filiere'));
    }

    public function configure(Filiere $filiere)
    {
        $filiere->load(['niveaux' => function($query) {
            $query->orderBy('id')->with(['matieres' => function($q) {
                $q->orderBy('ordre');
            }, 'frais', 'documentsNiveau' => function($q) {
                $q->orderBy('ordre');
            }]);
        }]);
        return view('admin.filieres.configure', compact('filiere'));
    }
}
