<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Gallery;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class GalleryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $images = Gallery::orderBy('created_at', 'desc')->get();
        return view('admin.galerie.index', compact('images'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.galerie.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:5120',
            'titre' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'statut' => 'boolean',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('galerie', 'public');
        }

        Gallery::create([
            'image_path' => $imagePath,
            'titre' => $request->titre,
            'description' => $request->description,
            'statut' => $request->has('statut') ? true : false,
        ]);

        return redirect()->route('admin.galerie.index')->with('success', 'Image ajoutée avec succès.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Gallery $galerie)
    {
        return view('admin.galerie.edit', compact('galerie'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Gallery $galerie)
    {
        $request->validate([
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
            'titre' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'statut' => 'boolean',
        ]);

        $imagePath = $galerie->image_path;

        if ($request->hasFile('image')) {
            // Supprimer l'ancienne image
            if ($galerie->image_path && Storage::disk('public')->exists($galerie->image_path)) {
                Storage::disk('public')->delete($galerie->image_path);
            }
            $imagePath = $request->file('image')->store('galerie', 'public');
        }

        $galerie->update([
            'image_path' => $imagePath,
            'titre' => $request->titre,
            'description' => $request->description,
            'statut' => $request->has('statut') ? true : false,
        ]);

        return redirect()->route('admin.galerie.index')->with('success', 'Image modifiée avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Gallery $galerie)
    {
        // Supprimer l'image du stockage
        if ($galerie->image_path && Storage::disk('public')->exists($galerie->image_path)) {
            Storage::disk('public')->delete($galerie->image_path);
        }

        $galerie->delete();

        return redirect()->route('admin.galerie.index')->with('success', 'Image supprimée avec succès.');
    }
}
